/* Copyright (C) 2002-2018 RealVNC Ltd. All Rights Reserved.
*/

#ifndef UUID_b67b2672_985b_4edf_81ec_984dc2a606b4
#define UUID_b67b2672_985b_4edf_81ec_984dc2a606b4

/**
 * \file vnccdbtypes.h
 *
 * Contains enumerations and types used with the CDB part of the VNC CDB and
 * SBP SDK API.
 *
 * The application should normally include vnccdbsdk.h rather than including
 * this file directly.
 *
 * \see vnccdbsdk.h, vnccdb.h
 */

#include "vncint.h"

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * \brief A CDB endpoint.
 *
 * Use VNCCDBEndpointCreate() to create a new CDB endpoint, and
 * VNCCDBEndpointDestroy() to destroy a CDB endpoint that is no longer
 * required.
 *
 * CDB endpoints created by VNCCDBEndpointCreate() may be used as CDB sources,
 * CDB sinks, or both at the same time.
 *
 * To use an endpoint as an SBP or CDB source, the application should add
 * services to the endpoint by calling VNCCDBEndpointAddSBPSource() or
 * VNCCDBEndpointAddCDBSource(). All services must be added before the endpoint
 * is started.
 *
 * After defining any services, the application should start the endpoint by
 * calling VNCCDBEndpointStart(), which begins the process of establishing the
 * CDB session. Once the session is established, the SDK will automatically
 * send a ServicesRequest message to the peer endpoint and will invoke the
 * ::VNCCDBEndpointServicesSupportedCallback when the peer endpoint responds
 * with a ServicesSupported message.
 *
 * To use an endpoint as a CDB sink, the application should call
 * VNCCDBEndpointAddSBPSink() or VNCCDBEndpointAddCDBSink(). These calls must
 * be made during or after the ::VNCCDBEndpointServicesSupportedCallback. The
 * application can then ask the peer endpoint to start these services by
 * calling VNCCDBEndpointSendStartService().
 *
 * To terminate the CDB session without destroying the endpoint, call
 * VNCCDBEndpointStop(). The endpoint may then be reused for further CDB
 * sessions.
 *
 * \see VNCCDBEndpointCreate(), VNCCDBEndpointDestroy(),
 * VNCCDBEndpointGetContext(), VNCCDBEndpointAddLicense(),
 * VNCCDBEndpointAddCDBSource(), VNCCDBEndpointAddCDBSink(),
 * VNCCDBEndpointStart(), VNCCDBEndpointStop(),
 * VNCCDBEndpointSendStartService(), VNCCDBEndpointSendStopService(),
 * VNCCDBEndpointSendServicePayload(), VNCCDBEndpointSendServiceResponse()
 */
typedef struct VNCCDBEndpoint VNCCDBEndpoint;

/**
 * \brief A CDB service ID.
 *
 * CDB services are identified by service IDs. The same service may have a
 * different service ID in different CDB sessions.
 *
 * \see VNCCDBServiceIsClientSide(), VNCCDBServiceIsServerSide()
 */
typedef vnc_uint16_t VNCCDBServiceId;

/**
 * \brief Indicates whether a CDB endpoint is for a MirrorLink client or a
 * MirrorLink server.
 *
 * \see ::VNCCDBEndpointDescriptor, VNCCDBEndpointCreate()
 */
typedef enum
{
    /** The endpoint is a client endpoint. */
    VNCCDBEndpointTypeClient = 1,
    /** The endpoint is a server endpoint. */
    VNCCDBEndpointTypeServer = 2
} VNCCDBEndpointType;

/**
 * \brief Used in the ::VNCCDBEndpointStatusCallback to indicate the current
 * status of a CDB endpoint.
 */
typedef enum
{
    /**
     * \brief The CDB endpoint is starting.
     *
     * The first action of the CDB thread is to provide this notification to
     * the application. This notification is therefore a good opportunity for
     * the application to allocate any data that is associated with the CDB
     * thread itself.
     *
     * The additionalInformation parameter to the
     * ::VNCCDBEndpointStatusCallback is not used.
     */
    VNCCDBEndpointStatusStarting = 1,

    /**
     * \brief The CDB endpoint has successfully established the TCP session to
     * the peer endpoint.
     *
     * The additionalInformation parameter to the
     * ::VNCCDBEndpointStatusCallback is not used.
     */
    VNCCDBEndpointStatusConnected = 2,

    /**
     * \brief The CDB endpoint is stopping.
     *
     * The last action of the CDB thread is to provide this notification to the
     * application. This notification is therefore a good opportunity for the
     * application to deallocate any data that is associated with the CDB
     * thread itself.
     *
     * The additionalInformation parameter may be cast to a ::VNCCDBError to
     * give the reason for the CDB thread exiting. If the CDB thread is exiting
     * because the application called VNCCDBEndpointStop(), then
     * additionalInformation will be ::VNCCDBErrorNone.
     */
    VNCCDBEndpointStatusStopping = 3,

    /**
     * \brief The CDB server endpoint has started listening for incoming
     * connections.
     */
    VNCCDBEndpointStatusListening = 4,

    /**
     * \brief The CDB client endpoint has started attemping to connect.
     */
    VNCCDBEndpointStatusConnecting = 5

} VNCCDBEndpointStatus;

/**
 * \brief Error codes that may be returned by SDK APIs.
 *
 * Error codes may be converted to strings by calling VNCCDBGetErrorName().
 *
 * More detailed information on any failures can often be found in the log. For
 * that reason, it is recommended that the application provide a
 * ::VNCCDBEndpointLogCallback.
 *
 * \see VNCCDBGetErrorName(), ::VNCCDBEndpointLogCallback
 */
typedef enum
{
    /**
     * \brief The operation was successful.
     */
    VNCCDBErrorNone                   = 0,

    /**
     * \brief The operation failed for an unspecified reason.
     *
     * Wherever possible, the SDK will attempt to provide a more specific
     * error code.
     */
    VNCCDBErrorFailed                 = 1,

    /**
     * \brief One or more of the parameters to an SDK API was invalid.
     */
    VNCCDBErrorInvalidParameter       = 2,

    /**
     * \brief A dynamic memory allocation failed.
     */
    VNCCDBErrorOutOfMemory            = 3,

    /**
     * \brief The desired protocol version is not supported by the SDK.
     */
    VNCCDBErrorUnsupportedVersion     = 4,

    /**
     * \brief A network error was returned by the OS.
     */
    VNCCDBErrorNetworkError           = 5,

    /**
     * \brief The application called an API that may not be called while the
     * CDB thread is running.
     */
    VNCCDBErrorIllegalWhileRunning    = 6,

    /**
     * \brief The application called an API that may not be unless the CDB
     * thread is running.
     */
    VNCCDBErrorIllegalWhileNotRunning = 7,

    /**
     * \brief The VNC license provided in the call to
     * VNCCDBEndpointAddLicense() is not valid.
     */
    VNCCDBErrorLicenseNotValid        = 8,

    /**
     * \brief The CDB endpoint cannot be started because it is not licensed.
     */
    VNCCDBErrorFeatureNotLicensed     = 9,

    /**
     * \brief There is no item that corresponds to a particular identifier
     * supplied as a parameter to an API.
     */
    VNCCDBErrorNotFound               = 10,

    /**
     * \brief The peer CDB endpoint sent invalid protocol.
     */
    VNCCDBErrorInvalidProtocol        = 11,

    /**
     * \brief The operation timed out.
     */
    VNCCDBErrorTimeout                = 12,

    /**
     * \brief An SBP error was detected by the SDK before any protocol was sent
     * to the peer endpoint.
     */
    VNCCDBErrorLocalSBPError          = 13,

    /**
     * \brief An SBP error was returned by the peer endpoint.
     */
    VNCCDBErrorRemoteSBPError         = 14,

    /**
     * \brief An invalid SBP packet was received.
     */
    VNCCDBErrorInvalidSBPPacket       = 15,

    /**
     * \brief Lowest error number that may be returned by SBP payload
     * serialization APIs.
     */
    VNCCDBErrorSBPSerialize               = 100,

    /**
     * \brief VNCSBPSerializeFinalize() was called before the content of the
     * ::VNCSBPSerialize object was completed (e.g. VNCSBPSerializeBeginArray()
     * was called, but there has been no corresponding call to
     * VNCSBPSerializeEnd().
     */
    VNCCDBErrorSBPSerializeIncomplete     = VNCCDBErrorSBPSerialize + 0,

    /**
     * \brief No further data can be added to the ::VNCSBPSerialize because it
     * is already complete.
     */
    VNCCDBErrorSBPSerializeComplete       = VNCCDBErrorSBPSerialize + 1,

    /**
     * \brief VNCSBPSerializeEnd() was called, but there was no corresponding
     * call to VNCSBPSerializeBeginArray() or VNCSBPSerializeBeginStructure().
     */
    VNCCDBErrorSBPSerializeUnexpectedEnd  = VNCCDBErrorSBPSerialize + 2,

    /**
     * \brief A non-zero UID was supplied when appending to a ::VNCSBPSerialize
     * object, but a UID cannot be used here, because the data is being
     * appended to an array (array members do not have UIDs).
     */
    VNCCDBErrorSBPSerializeUnexpectedUID  = VNCCDBErrorSBPSerialize + 3,

    /**
     * \brief Data of the given type may not be appended at this time, because
     * the data is being appended to a container whose members must be of a
     * different type (e.g. VNCSBPSerializeInt() was called, but the container
     * is an array of Booleans).
     */
    VNCCDBErrorSBPSerializeUnexpectedType = VNCCDBErrorSBPSerialize + 4,

    /**
     * \brief Lowest error number that may be returned by SBP payload
     * deserialization APIs.
     */
    VNCCDBErrorSBPDeserialize = 200,

    /**
     * \brief The data being deserialized contains data of a type whose
     * corresponding callback was not supplied in the
     * ::VNCSBPDeserializeCallbacks.
     */
    VNCCDBErrorSBPDeserializeCallbackNotImplemented =
        VNCCDBErrorSBPDeserialize + 0,

    /**
     * \brief The data being deserialized is too short and does not represent a
     * complete SBP payload.
     */
    VNCCDBErrorSBPDeserializeInputTruncated =
        VNCCDBErrorSBPDeserialize + 1,

    /**
     * \brief The data being deserialized contains a complete SBP payload, and
     * was deserialized successfully, but there is extra data at the end.
     */
    VNCCDBErrorSBPDeserializeInputTooLong =
        VNCCDBErrorSBPDeserialize + 2,

    /**
     * \brief The data being deserialized is not a valid SBP payload.
     */
    VNCCDBErrorSBPDeserializeInputMalformed =
        VNCCDBErrorSBPDeserialize + 3
} VNCCDBError;

/**
 * \brief Service configuration values.
 *
 * See section 3.2 of \ref ml_cdb "[ML-CDB]" for an explanation.
 *
 * \see ::VNCCDBService
 */
typedef enum
{
    /**
     * \brief Empty value.
     */
    VNCCDBServiceConfigurationNone               = 0x00000000,

    /**
     * \brief ServicePayload message payloads are encrypted by the service
     * provider.
     *
     * It is worthwhile to be noted that the encryption/decryption must be
     * done by the service provider/consumer and the SDK itself doesn't
     * encrypt/decrypt the payload if the following value is set.
     *
     * Known Issue:
     * In CCC-TS-016 v1.1 and v1.2, section 'ServicePayload' requires that the
     * service access control byte included in each service payload message
     * must be encrypted as well if the following value is set. But the service
     * access control byte is treated as a CDB protocol header of the service
     * payload message in the SDK and there is no way for the service providers
     * to encrypt the service access control byte in the service payload message.
     */
    VNCCDBServiceConfigurationEncryption         = 0x00000001,

    /**
     * \brief If the CDB endpoint hosts multiple services with the same name,
     * but different version numbers, then only one of the similarly-named
     * services may be started.
     */
    VNCCDBServiceConfigurationResourceConstraints = 0x00000002,
} VNCCDBServiceConfiguration;

/**
 * \brief Service access control values.
 *
 * See section 3.2 of \ref ml_cdb "[ML-CDB]" for an explanation.
 *
 * \see ::VNCCDBService
 */
typedef enum
{
    /**
     * \brief Empty value.
     */
    VNCCDBServiceAccessControlNone               = 0x00000000,

    /**
     * \brief Data can be provided to any application.
     */
    VNCCDBServiceAccessControlUnlimited          = 0x00000001,

    /**
     * \brief Data can be provided to any application that has been certified
     * by the Car Connectivity Consortium.
     */
    VNCCDBServiceAccessControlCCCCertified       = 0x00000002,

    /**
     * \brief Data can be provided to any application that has been certified
     * by the manufacturer of the CDB source device.
     */
    VNCCDBServiceAccessControlSourceCertified    = 0x00000004,

    /**
     * \brief Data can be provided to any application that has been certified
     * by the manufacturer of the CDB source device as being permitted to
     * access this service.
     */
    VNCCDBServiceAccessControlServiceCertified   = 0x00000008,
} VNCCDBServiceAccessControl;

/**
 * \brief Describes a CDB service.
 *
 * This structure is used by:
 *
 *  - VNCCDBEndpointAddCDBSource(), to describe the service that is being
 *    added.
 *  - ::VNCCDBEndpointServicesSupportedCallback, to describe the services that
 *    are supported by the peer CDB endpoint.
 *  - ::VNCCDBEndpointServicesNotSupportedCallback, to describe the services
 *    that are no longer supported by the peer CDB endpoint.
 *
 * \see VNCCDBEndpointAddCDBSource(),
 * ::VNCCDBEndpointServicesSupportedCallback,
 * ::VNCCDBEndpointServicesNotSupportedCallback, ::VNCCDBServiceId,
 * ::VNCCDBServiceConfiguration, ::VNCCDBServiceAccessControl
 */
typedef struct
{
    /**
     * \brief The service ID.
     *
     * When used with VNCCDBEndpointAddCDBSource(), the value of this field is
     * overwritten by the SDK, which will assign a new, unused service ID to
     * the service and return it in this field.
     *
     * When used with ::VNCCDBEndpointServicesSupportedCallback, the value of
     * this field is the service ID assigned to the service by the peer
     * endpoint.
     */
    VNCCDBServiceId serviceId;

    /**
     * \brief The service name.
     *
     * As per \ref ml_cdb "[ML-CDB]", CDB service names should follow the
     * reverse domain name convention.
     */
    const char *name;

    /** The major component of the service's version number. */
    vnc_uint8_t versionMajor;

    /** The minor component of the service's version number. */
    vnc_uint8_t versionMinor;

    /** Service configuration value. */
    VNCCDBServiceConfiguration configuration;

    /** Service access control value. */
    VNCCDBServiceAccessControl accessControl;
} VNCCDBService;

/**
 * \brief CDB ServiceResponse codes.
 *
 * These are provided to the application in the
 * ::VNCCDBEndpointServiceResponseCallback associated with a CDB sink.
 *
 * See section 3.6 of \ref ml_cdb "[ML-CDB]" for an explanation.
 *
 * \see ::VNCCDBEndpointServiceResponseCallback, VNCCDBEndpointAddCDBSink()
 */
typedef enum
{
    /**
     * \brief Empty value.
     */
    VNCCDBServiceResponseNone                           = 0x0000,

    /**
     * \brief The service was started successfully.
     */
    VNCCDBServiceResponseOKServiceStarted               = 0x0001,

    /**
     * \brief The service was stopped successfully.
     */
    VNCCDBServiceResponseOKServiceStopped               = 0x0002,

    /**
     * \brief A ServiceResponse is still pending.
     */
    VNCCDBServiceResponsePending                        = 0x0010,

    /**
     * \brief The service was already running, and an attempt was made to start
     * it.
     */
    VNCCDBServiceResponseWarningServiceRunning          = 0x0101,

    /**
     * \brief The service was not running, and an attempt was made to stop it.
     */
    VNCCDBServiceResponseWarningServiceNotRunning       = 0x0102,

    /**
     * \brief The service could not be started.
     */
    VNCCDBServiceResponseErrorServiceLaunchFailed       = 0x0201,

    /**
     * \brief The service could not be stopped.
     */
    VNCCDBServiceResponseErrorServiceTerminationFailed  = 0x0202,

    /**
     * \brief The service was stopped by its source endpoint. If desired, the
     * sink endpoint can restart it.
     */
    VNCCDBServiceResponseErrorServiceReset              = 0x0203,

    /**
     * \brief The service was stopped by its source endpoint. It is now
     * unavailable and cannot be restarted.
     */
    VNCCDBServiceResponseErrorServiceTerminated         = 0x0204,

    /**
     * \brief The peer endpoint does not recognize a service ID that was sent
     * to it in a StartService, StopService or ServicePayload message.
     */
    VNCCDBServiceResponseErrorUnknownServiceId          = 0x0205,

    /**
     * \brief The service access control value that was supplied in a
     * ServicePayload message is incorrect.
     */
    VNCCDBServiceResponseErrorWrongAccessControl        = 0x0206,

    /**
     * \brief The service cannot be started because it is resource-constrained
     * and another version of the service is running.
     */
    VNCCDBServiceResponseErrorResourceBusy              = 0x0207,

    /**
     * \brief The peer endpoint received a PingResponse containing the wrong
     * Ping sequence number.
     */
    VNCCDBServiceResponseErrorWrongSequenceNumber       = 0x0208,

    /**
     * \brief The peer endpoint does not support the requested version of the
     * CDB protocol.
     */
    VNCCDBServiceResponseErrorWrongCDBVersionNumber     = 0x0209,

    /**
     * \brief The service does not support service version number requested in
     * the StartService message.
     */
    VNCCDBServiceResponseErrorWrongServiceVersionNumber = 0x020A,

    /**
     * \brief The service does not recognize the format of the payload data in
     * a ServicePayload message.
     */
    VNCCDBServiceResponseErrorUnsupportedPayloadFormat  = 0x020B,
} VNCCDBServiceResponse;


#ifdef __cplusplus
}
#endif

#endif /* !defined(UUID_b67b2672_985b_4edf_81ec_984dc2a606b4) */
